#version 430 core
layout (location = 0) out vec4 lightingAlbedoSpec;

in vec2 TexCoords;

layout (binding = 0) uniform sampler2D gAlbedoSpec; // this is a vec4
layout (binding = 1) uniform sampler2D gViewPosition; //this is a vec3
layout (binding = 2) uniform sampler2D gNormal; //this is a vec3
layout (binding = 3) uniform sampler2D blurBloomShaderOutput; //vec3, downsampled for better bloomage
layout (binding = 4) uniform sampler2D gShadowVisibility;
layout (binding = 5) uniform sampler2D ssaoOcclusion; // this is a float, used for SSAO
layout (binding = 6) uniform sampler2D gPhysical; 

layout (binding = 7) uniform sampler2D bAlbedoSpec; // this is a float
layout (binding = 8) uniform sampler2D bViewPosition;
layout (binding = 9) uniform sampler2D bNormal; // this is also a vec4
layout (binding = 10) uniform sampler2D bPhysical; 

layout (binding = 11) uniform sampler2D pAlbedoAlpha; // this is also a vec4
layout (binding = 12) uniform sampler2D distortionTex;
layout (binding = 13) uniform sampler2D quadAlbedoAlpha; //vec4, with alpha being the alpha component
layout (binding = 14) uniform sampler2D cloudAlbedoAlpha; //vec4, with alpha being the alpha component
layout (binding = 15) uniform sampler2D solarAlbedoAlpha;
layout (binding = 16) uniform sampler2D godraysAlbedoAlpha;

const int NR_LIGHTS = 2; 

struct Light {
  vec4 PositionLinear;
  vec4 ColorQuadratic;
	vec4 IntensityPADDING_PADDING_PADDING;
};

layout (std140, binding = 0) uniform LightBlock {
  Light lights[NR_LIGHTS];
};

uniform int lightCount;

uniform int areShadowsOn;
uniform int isSSAOn;

uniform int quadToRender;
uniform float monochromeRatio;
uniform float ambientStrength;

uniform vec3 viewDir;
uniform vec2 mousePos;
uniform mat4 view;

layout(r32i, binding = 1) uniform iimage2D atomicBloomTex;

uniform int SCR_WIDTH;
uniform int SCR_HEIGHT;


//functions for PBR

const float PI = 3.14159265359;

float DistributionGGX(vec3 N, vec3 H, float roughness)
{
  float a = roughness*roughness;
  float a2 = a*a;
  float NdotH = max(dot(N, H), 0.0);
  float NdotH2 = NdotH*NdotH;

  float nom  = a2;
  float denom = (NdotH2 * (a2 - 1.0) + 1.0);
  denom = PI * denom * denom;

  return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySchlickGGX(float NdotV, float roughness)
{
  float r = (roughness + 1.0);
  float k = (r*r) / 8.0;

  float nom  = NdotV;
  float denom = NdotV * (1.0 - k) + k;

  return nom / denom;
}
// ----------------------------------------------------------------------------
float GeometrySmith(vec3 N, vec3 V, vec3 L, float roughness)
{
  float NdotV = max(dot(N, V), 0.0);
  float NdotL = max(dot(N, L), 0.0);
  float ggx2 = GeometrySchlickGGX(NdotV, roughness);
  float ggx1 = GeometrySchlickGGX(NdotL, roughness);

  return ggx1 * ggx2;
}
// ----------------------------------------------------------------------------
vec3 fresnelSchlick(float cosTheta, vec3 F0)
{
  return F0 + (1.0 - F0) * pow(1.0 - cosTheta, 5.0);
}


float blurShadow(vec2 texCoordsToUse){

  float result = texture(gShadowVisibility, texCoordsToUse).r;

	vec2 texOffsetMagnitude = vec2(1.0 / float(SCR_WIDTH), 1.0 / float(SCR_HEIGHT)) * 0.5;

	result += texture(gShadowVisibility, texCoordsToUse + texOffsetMagnitude * vec2(0.0, 1.0)).r * 0.25;
	result += texture(gShadowVisibility, texCoordsToUse + texOffsetMagnitude * vec2(0.0, -1.0)).r * 0.25;
	result += texture(gShadowVisibility, texCoordsToUse + texOffsetMagnitude * vec2(1.0, 0.0)).r * 0.25;
	result += texture(gShadowVisibility, texCoordsToUse + texOffsetMagnitude * vec2(-1.0, 0.0)).r * 0.25;

	return (result / 1.5f);
}


float blurSSAO(vec2 texCoordsToUse){
	float result = texture(ssaoOcclusion, texCoordsToUse).r;

	vec2 texOffsetMagnitude = vec2(1.0 / float(SCR_WIDTH), 1.0 / float(SCR_HEIGHT));

	result += texture(ssaoOcclusion, texCoordsToUse + texOffsetMagnitude * vec2(0.0, 1.0)).r * 0.5;
	result += texture(ssaoOcclusion, texCoordsToUse + texOffsetMagnitude * vec2(0.0, -1.0)).r * 0.5;
	result += texture(ssaoOcclusion, texCoordsToUse + texOffsetMagnitude * vec2(1.0, 0.0)).r * 0.5;
	result += texture(ssaoOcclusion, texCoordsToUse + texOffsetMagnitude * vec2(-1.0, 0.0)).r * 0.5;

	return (result / 3.0f);
}

vec4 blurGodRays(vec2 texCoordsToUse){
	vec4 result = texture(godraysAlbedoAlpha, texCoordsToUse).rgba;

	vec2 texOffsetMagnitude = vec2(1.0 / float(SCR_WIDTH), 1.0 / float(SCR_HEIGHT));

	result += texture(godraysAlbedoAlpha, texCoordsToUse + texOffsetMagnitude * vec2(0.0, 1.0)).rgba * 0.5;
	result += texture(godraysAlbedoAlpha, texCoordsToUse + texOffsetMagnitude * vec2(0.0, -1.0)).rgba * 0.5;
	result += texture(godraysAlbedoAlpha, texCoordsToUse + texOffsetMagnitude * vec2(1.0, 0.0)).rgba * 0.5;
	result += texture(godraysAlbedoAlpha, texCoordsToUse + texOffsetMagnitude * vec2(-1.0, 0.0)).rgba * 0.5;

	return (result / 3.0f);
}


vec4 blurCloud(vec2 texCoordsToUse){
	vec4 result = texture(cloudAlbedoAlpha, texCoordsToUse).rgba;

	return (result);
}


uniform float exposure = 2.0;
uniform float gamma = 1.0;


vec3 computeLighting(vec3 FragPos, vec3 FragNormal, vec3 albedo, float roughness, float SSAOAmount, float shadowVisibility, bool hasANormal){

		if(hasANormal == false){
			return albedo;
		}

		float metallic = 1.0 - roughness;

		vec3 F0 = vec3(0.04); 
		F0 = mix(F0, albedo, metallic);

		//ambient 

		vec3 ambient = vec3(ambientStrength * (1.0 + SSAOAmount));
		vec3 lighting= vec3(0.0);

		vec3 V = normalize(viewDir);
		vec3 N = normalize(FragNormal);
		
		//diffuse term
		for(int i = 0; i < lightCount; i++){

    		//shadows happen here
			float visibility = 1.0;

			if(i == 0){		
				visibility = shadowVisibility;			
			}

			// calculate per-light radiance
			vec3 L = normalize(lights[i].PositionLinear.xyz - FragPos);
			vec3 H = normalize(V + L);
			float distance = length(lights[i].PositionLinear.xyz - FragPos);
			float attenuation = (1.0) / (1.0 + lights[i].PositionLinear.w * distance + lights[i].ColorQuadratic.w * distance * distance);
			vec3 radiance = lights[i].ColorQuadratic.xyz * attenuation;

			// Cook-Torrance BRDF
			float NDF = DistributionGGX(N, H, roughness);  
			float G  = GeometrySmith(N, V, L, roughness);   
			vec3 F  = fresnelSchlick(max(dot(H, V), 0.0), F0);
      
			vec3 nominator  = NDF * G * F; 
			float denominator = 4 * max(dot(N, V), 0.0) * max(dot(N, L), 0.0) + 0.001; // 0.001 to prevent divide by zero.
			vec3 specular = nominator / denominator;
    
			// kS is equal to Fresnel
			vec3 kS = F;
			// for energy conservation, the diffuse and specular light can't
			// be above 1.0 (unless the surface emits light); to preserve this
			// relationship the diffuse component (kD) should equal 1.0 - kS.
			vec3 kD = vec3(1.0) - kS;
			// multiply kD by the inverse metalness such that only non-metals 
			// have diffuse lighting, or a linear blend if partly metal (pure metals
			// have no diffuse light).
			kD *= 1.0 - metallic;	 

			// scale light by NdotL
			float NdotL = max(dot(N, L), 0.0);    


			// add to outgoing radiance Lo
			lighting += (kD / PI + specular) * radiance * NdotL * visibility * lights[i].IntensityPADDING_PADDING_PADDING.x; // note that we already multiplied the BRDF by the Fresnel (kS) so we won't multiply by kS again

		}

     
		return (lighting + ambient) * albedo;

}


void main()
{      

	float distortionVector = texture(distortionTex, TexCoords).a;

	vec2 texCoordsToUse = TexCoords;

	if(distortionVector > 0){
		texCoordsToUse -= texture(distortionTex, TexCoords).xy * distortionVector / 10.0f;
	}
		
	vec3 hdrColor;

	vec4 blendPhysical = texture(bPhysical, texCoordsToUse).xyzw;

	vec3 solidFragPos;
	vec3 solidFragNormal;
	vec3 solidAlbedo;
	float solidRoughness;

	vec3 blendFragPos;
	vec3 blendFragNormal;
	vec3 blendAlbedo;
	float blendRoughness;

	solidFragNormal = texture(gNormal, texCoordsToUse).rgb;
	solidAlbedo = texture(gAlbedoSpec, texCoordsToUse).rgb;
	blendFragNormal = texture(bNormal, texCoordsToUse).rgb;

	vec4 _cloudAlbedoAlpha = blurCloud(texCoordsToUse); //some tuning can be done here I think

	 vec4 particleAlbedoAlpha = texture(pAlbedoAlpha, texCoordsToUse).rgba;

	 float visibility = 1.0;

	//check to see if skybox

	if(length(blendFragNormal) == 0.0f && length(solidFragNormal) == 0.0f && _cloudAlbedoAlpha.a == 0.0f && particleAlbedoAlpha.a == 0.0f){

		hdrColor = solidAlbedo;

	}else{
			
		float alpha = 1.0 - blendPhysical.w;

		alpha = clamp(alpha, 0.0, 1.0);
					
			solidFragPos = texture(gViewPosition, texCoordsToUse).rgb;
			solidRoughness = texture(gPhysical, texCoordsToUse).r;

			blendFragPos = texture(bViewPosition, texCoordsToUse).rgb;
			blendAlbedo = texture(bAlbedoSpec, texCoordsToUse).rgb;
			blendRoughness = texture(bPhysical, texCoordsToUse).r;

			float SSAOAmount = 1.0;

			if(isSSAOn == 1){
				SSAOAmount = blurSSAO(texCoordsToUse) * alpha;
			}

			if(areShadowsOn == 1){		
				visibility = blurShadow(texCoordsToUse);
			}

			vec3 solidPart;
			vec3 blendPart;

			if(alpha != 0.0){

				bool hasANormal = length(solidFragNormal) != 0.0;

				solidPart = computeLighting(solidFragPos, solidFragNormal, solidAlbedo, solidRoughness, SSAOAmount, visibility, hasANormal);
			}

			if(alpha != 1.0){
				blendPart = computeLighting(blendFragPos, blendFragNormal, blendAlbedo, blendRoughness, 1.0, visibility, true);
			}

					if(alpha != 1.0){
			hdrColor = mix(solidPart, blendPart, 1.0 - alpha);
			}else{
				hdrColor = solidPart;
			}


			//add particles here:
		

			if(particleAlbedoAlpha.a != 0.0){

				hdrColor = (hdrColor * (1.0 - particleAlbedoAlpha.a)) + (particleAlbedoAlpha.rgb * particleAlbedoAlpha.a);

			}else{
				
				hdrColor = hdrColor;

			}

		}


		//Clouds
		float t = pow(_cloudAlbedoAlpha.a, 2);
		hdrColor = mix(hdrColor, _cloudAlbedoAlpha.rgb, t);
					
		//GodRays
		vec4 godrayInput = blurGodRays(texCoordsToUse);
		hdrColor = mix(hdrColor, godrayInput.rgb, godrayInput.a);

		//solar
		vec4 solarInput = texture(solarAlbedoAlpha, texCoordsToUse).rgba;
		hdrColor = mix(hdrColor, solarInput.rgb, solarInput.a);

		//Bloom

		vec4 bloomOut = texture(blurBloomShaderOutput, texCoordsToUse).rgba;
		float bloomStrength = bloomOut.a;

		if(bloomStrength > 0.0){

			bloomStrength = clamp(bloomStrength, 0.0, 1.0);

			hdrColor = mix(hdrColor, bloomOut.rgb, bloomStrength);
		}

		// Exposure tone mapping
		vec3 mapped = vec3(1.0) - exp(-hdrColor * exposure);
		// Gamma correction 
		mapped = pow(mapped, vec3(1.0 / gamma));
	
		lightingAlbedoSpec = vec4(mapped, 1.0);

		//lightingAlbedoSpec = vec4(texture(blurBloomShaderOutput, texCoordsToUse).rgb, 1.0);

	 if(monochromeRatio > 0.0){
		
		float gray = dot(lightingAlbedoSpec.rgb, vec3(0.299, 0.587, 0.114));
		lightingAlbedoSpec.rgb = vec3(gray) * monochromeRatio + lightingAlbedoSpec.rgb * (1.0 - monochromeRatio);
	 }
	 
	 if(quadToRender == 1){
		float quadAlpha = texture(quadAlbedoAlpha, TexCoords).a;
		vec3 quadColor = texture(quadAlbedoAlpha, TexCoords).rgb;

		lightingAlbedoSpec.rgb = ((1.0 - quadAlpha) * lightingAlbedoSpec.rgb) + (quadColor * quadAlpha);
	 }

}
